<?php
defined('BASEPATH') OR exit('No direct script access allowed');
/**
 * Razorpay API Ref: 	https://github.com/razorpay/razorpay-php
 * Payu API Ref: 		https://www.payumoney.com/pdf/API-Docs.pdf
 * Paypal API Ref: 		https://developer.paypal.com/docs/checkout/integration-features/refunds/#refunds-with-the-payments-api
 *Stripe API Ref:		https://stripe.com/docs/api/refunds/create
 * @author Baji Shaik <bajidotwork@gmail.com>
 * @version 1.1 24th Aug 2019
 */

require_once APPPATH. '/third_party/razorpay/Razorpay.php';
require_once APPPATH. '/third_party/stripe_package/init.php';
use Razorpay\Api\Api;

/*------------Online Refunds For Dattapeetham Online Orders (Paypal, Payu, Razorpay, Stripe)-----*/
class Onlinerefund
{
	protected $CI;

	public function __construct()
	{
        $this->CI =& get_instance();
        $this->CI->load->model('orders/orders_model');
        $this->CI->load->model('ashrams/ashrams_trust_accounts_model');
	}

	/**
	 * Process Refund Based On Payment Mode
	 * @param  String 	$payment_mode [Order Payment Mode]
	 * @param  Array 	$tx_details   [Transaction details Row]
	 * @param  Array 	$orderData    [Order details Row]
	 * @return Array               	  [Gateway Refund Response]
	 */
	public function process_refund($payment_mode,$tx_details,$orderData,$amount=0)
	{
		$ashramId 		= $tx_details->ashram_id;
		$trustId 		= $tx_details->trust_id;
		$cartType 		= $orderData['order']->order_type;
		$tx_currency 	= $tx_details->currency;
		$return 		= false;

		if(!empty($amount) && $amount > 0)
		{
			$amount = $amount;
		}
		else
		{
			$amount = $tx_details->amount;
		}

		// $trustData 		= $this->getTrustData($ashramId,$cartType,$tx_currency);
		$trustData = $this->CI->ashrams_trust_accounts_model->trustData($trustId);
		if(empty($trustData))
		{
			$return = array("status"=>2,"response"=>"","message"=>"Refund not processed, please check your trust configuration details");
		}
		if(strtolower($payment_mode) == "razorpay")
		{
			$return = $this->razorpayRefund($tx_details,$trustData,$amount);
		}
		if(strtolower($payment_mode) == "paypal")
		{
			$return = $this->paypalRefund($tx_details,$trustData,$amount);
		}
		if(strtolower($payment_mode) == "payu")
		{
			$return = $this->payuRefund($tx_details,$trustData,$amount);
		}
		if(strtolower($payment_mode) == "stripe")
		{
			$return = $this->stripeRefund($tx_details,$trustData,$amount);
		}
		return $return;
	}

	//Get Trust data eg: Gateway Keys
	private function getTrustData($ashramId,$cartType,$tx_currency)
	{
		if(!empty($ashramId) && !empty($cartType) && !empty($tx_currency))
		{
			$trusts=$this->CI->ashrams_trust_accounts_model->getTrustsByAshramId($ashramId,$cartType);
			$trustsArray=array();
	    	foreach ($trusts as $trustData)
	    	{
	    		$trustsArray[$trustData->currency]=$trustData;
	    	}
	    	$trustData=$trustsArray[strtolower($tx_currency)];
	    	return $trustData;
		}
		else
		{
			return array("status"=>2,"response"=>"","message"=>"Refund not processed, please check your trust configuration details");
		}
	}

	/**
	 * [razorpayRefund Full Refund 	Tx Level]
	 * @param  [array] $tx_details 	Transaction Row
	 * @param  [array] $trustData  	Trust Configuration Details
	 * @return [array]            	Status with response
	 */
	private function razorpayRefund($tx_details,$trustData=null,$amount)
	{
		if(empty($trustData->razorpay_key) || empty($trustData->razorpay_secret))
		{
			return array("status"=>2,"response"=>"","message"=>"Refund not processed, please check your trust configuration details");
		}
		try
		{
			$api = new Api($trustData->razorpay_key, $trustData->razorpay_secret);		
			// Creates refund for a payment
			$amount = $amount*100;
			$refund = $api->refund->create(array('payment_id' => $tx_details->tx,'amount'=>$amount));		
			return array("status"=>1,"response"=>$refund,"message"=>"Refund processed successfully");
		}
		catch(Exception $e)
		{
			return array("status"=>3,"response"=>"","message"=>$e->getMessage());
		}
	}

	/**
	 * [stripeRefund Full Refund 	Tx Level]
	 * @param  [array] $tx_details 	Transaction Row
	 * @param  [array] $trustData  	Trust Configuration Details
	 * @return [array]            	Status with response
	 */
	private function stripeRefund($tx_details,$trustData=null,$amount)
	{
		if(empty($trustData->stripe_secret_key))
		{
			return array("status"=>2,"response"=>"","message"=>"Refund Not Processed, Please Check Your Trust Configuration Details");
		}
		try
		{
			\Stripe\Stripe::setApiKey($trustData->stripe_secret_key);
			$refund = \Stripe\Refund::create([
			  	'charge' => $tx_details->tx,
			  	'amount'=>$amount*100,
			]);

			return array("status"=>1,"response"=>$refund,"message"=>"Refund Processed Successfully");
		}
		catch(Exception $e)
		{
			return array("status"=>3,"response"=>"","message"=>$e->getMessage());
		}
	}

	/**
	 * [payuRefund Full Refund 	Tx Level]
	 * @param  [array] $tx_details 	Transaction Row
	 * @param  [array] $trustData  	Trust Configuration Details
	 * @return [array]            	Status with response
	 */
	private function payuRefund($tx_details,$trustData=null,$amount)
	{
		if(empty($trustData))
		{
			return array("status"=>2,"response"=>"","message"=>"Refund not processed, please check your trust configuration details");
		}

		$headers = array();
		$headers[] = 'Content-Type: application/json;';
		$headers[] = 'Authorization: '.$trustData->payu_auth_key;

		$get_payment_url = $trustData->payu_get_url."/payment/op/getPaymentResponse";
		$p_api_request_params['merchantKey']  = $trustData->payu_merchantkey;
		$p_api_request_params['merchantTransactionIds']    = $tx_details->tx;
		$paymentData = $this->buildPayuCurl($p_api_request_params,$get_payment_url,$headers);
		if(!empty($paymentData) && $paymentData->status == "0")
		{
			foreach ($paymentData->result as $key => $value) 
			{
				$paymentId = $value->postBackParam->paymentId;
				$amount = $value->postBackParam->amount;
			}
			$post_refund_url = $trustData->payu_get_url."/treasury/merchant/refundPayment";
			$r_api_request_params['merchantKey'] 	= $trustData->payu_merchantkey;
			$r_api_request_params['paymentId'] 		= $paymentId;
			$r_api_request_params['refundAmount'] 	= $amount;
			$r_api_request_params['refundToSource'] = "";
			$refundData = $this->buildPayuCurl($r_api_request_params,$post_refund_url,$headers);
			if(!empty($refundData) && $refundData->status == "0")
			{
				return array("status"=>1,"response"=>$refundData,"message"=>"Refund processed successfully");
			}
			else
			{
				return array("status"=>3,"response"=>"","message"=>$refundData->message);
			}
		}
		else
		{
			return array("status"=>3,"response"=>"","message"=>$paymentData->message);
		}
	}

	/**
	 * [paypalRefund Full Refund 	Tx Level]
	 * @param  [array] $tx_details 	Transaction Row
	 * @param  [array] $trustData  	Trust Configuration Details
	 * @return [array]            	Status with response
	 */
	private function paypalRefund($tx_details,$trustData=null,$amount)
	{
		if(empty($trustData))
		{
			return array("status"=>2,"response"=>"","message"=>"Refund not processed, please check your trust configuration details");
		}
		
		if(trim(@$tx_details->currency)=="")
		{
			return array("status"=>3,"response"=>"","message"=>"Refund not processed, currency code is missing");
		}
        
        if(trim(@$tx_details->tx)=="")
        {
        	return array("status"=>3,"response"=>"","message"=>"Refund not processed, Tx ID is missing");
        }
        
        if(empty($amount))
        {
            return array("status"=>3,"response"=>"","message"=>"Refund not processed, amount is missing");
        }

		$DataInArray = array();
	    $DataInArray['transactionID'] 	= $tx_details->tx;
	    $DataInArray['refundType'] 		= "Full"; //Partial or Full
	    $DataInArray['currencyCode'] 	= strtoupper($tx_details->currency);
	    $DataInArray['amount'] 			= $amount;
	    $DataInArray['memo']			= "Full Refund For ".$tx_details->tx;
	    $DataInArray['invoiceID'] 		= "";
	    $aryRes = $this->buildPaypalRefundCurl($DataInArray,$trustData);
	    if($aryRes['RESPMSG'] == "Approved")
	    {         
	    	return array("status"=>1,"response"=>$aryRes,"message"=>"Refund processed successfully");
	    }
	    else
	    {
	    	return array("status"=>3,"response"=>"","message"=>$aryRes['RESPMSG']);
	    }
	}

	/**
	 * [buildPaypalRefundCurl description]
	 * @param  [type] $DataInArray [description]
	 * @param  [type] $trustData   [description]
	 * @return [type]              [description]
	 */
	private function buildPaypalRefundCurl($DataInArray,$trustData)
	{
		$api_request_params = array (
		    			// API Data
		    			'METHOD'=>'RefundTransaction',
		    			'USER' => $trustData->payflow_user,
                        'VENDOR' => $trustData->payflow_vendor,
                        'PARTNER' => $trustData->payflow_partner,
                        'PWD' => $trustData->payflow_pwd,
                        'TENDER' => 'C', // C = credit card, P = PayPal
						'TRXTYPE' => 'C', //  S=Sale, A= Auth, C=Credit, D=Delayed Capture, V=Void                        
						'ORIGID' => $DataInArray['transactionID'],
						'AMT' => $DataInArray['amount'],
						'CURRENCY' => $DataInArray['currencyCode']
		    	);

        if(trim(@$DataInArray['invoiceID'])!="")
            $api_request_params['INVOICEID'] = $DataInArray['invoiceID'];

        if(isset($DataInArray['memo']))
            $api_request_params['NOTE'] = $DataInArray['memo'];

    	$nvp = $this->toNVP($api_request_params);

    	$curl = curl_init();
    	curl_setopt($curl, CURLOPT_VERBOSE, 1);
    	curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, FALSE);
    	curl_setopt($curl, CURLOPT_TIMEOUT, 30);
    	curl_setopt($curl, CURLOPT_URL, $trustData->payflow_url);
    	curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
    	curl_setopt($curl, CURLOPT_POSTFIELDS, $nvp);
    	$curlResponse = curl_exec($curl);
        if(!$curlResponse)
        {
        	return array("status"=>3,"response"=>"","message"=>"RefundTransaction failed".curl_error($curl_var)."(".curl_errno($curl_var).")");
        }
        //RESULT=0&PNREF=B52P0B3C4645&RESPMSG=Approved&PPREF=0H08939299576181Y&CORRELATIONID=9ad5d511f5ae
        // Extract the response details.
        $httpResponseAr = explode("&", $curlResponse);

        $aryResponse = array();
        foreach ($httpResponseAr as $i => $value)
        {
            $tmpAr = explode("=", $value);
            if(sizeof($tmpAr) > 1)
            {
                $aryResponse[$tmpAr[0]] = urldecode($tmpAr[1]);
            }
        }
        if(0 == sizeof($aryResponse))
        {
        	return array("status"=>3,"response"=>"","message"=>"Invalid HTTP response for POST request ($nvp) to ".$trustData->payflow_url);
        }
        return $aryResponse;
	}

	public function buildPayuCurl(array $api_request_params,$url,array $headers)
	{
		// Building query string using http_build_query, because this API having parameters to pass in query as GET.
		// If not used http_query_build it will throw 400 "Bad Request".
		$defaults = array( 
		        CURLOPT_URL => $url. (strpos($url, '?') === FALSE ? '?' : ''). http_build_query($api_request_params), 
		        CURLOPT_HEADER => 0, 
		        CURLOPT_HTTPHEADER => $headers, 
		        CURLOPT_RETURNTRANSFER => TRUE, 
		        CURLOPT_TIMEOUT => 400
		    );
		// Initializing cURL    
		$ch = curl_init();
		curl_setopt_array($ch, ($defaults));
		curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
		if(!$result = curl_exec($ch)) 
		{ 
		    $error = trigger_error(curl_error($ch)); 
		} 
		curl_close($ch); 
		$resultarray=$result; 
		return json_decode($resultarray);
	}

	//To Build Query String
	private function toNVP($array)
	{
    	$i = 0;
    	$nvp = "";
    	foreach($array as $key => $val)
    	{
    		if($i != 0)
    		{
    			$nvp .= "&";
    		}
    		$nvp .= $key . '=' . $val;
    		$i++;
    	}
    	return $nvp;
    }
}
/* End of file Onlinerefund.php */
/* Location: ./application/libraries/Onlinerefund.php */